% Copyright (c) 2011 Joris Degroote
%
% Licensed under the Non-Profit Open Software License version 3.0
% (NPOSL-3.0). You should have received a copy of the NPOSL-3.0 along with 
% this file. If not, see <http://www.opensource.org/licenses/NPOSL-3.0>
%
% Calculation of the fluid flow in a tube.

classdef Fluid < handle
    properties (Constant)
        mltom3=1.0e-6;
        mmHgtoPa=133.322;
        r_o=0.005;
        h=0.003;
        mu=0.003;
        HR=1.0;
        rho=1000;
        sca=1.0e8;
    end
    properties (SetAccess=immutable)
        dt;
        dz;
        m_e;
        m_fo;
        m_fg;
        m_fh;
        m_sg;
        ell;
        C_d;
        C_po;
        C_p;
        R;
        L;
        p_v;
        SV;
        T_b;
        T_s;
        QA;
        QB;
        
        C_f;
        D_f;
        M_f;
        N_f;
    end
    properties (SetAccess=private)
        n;
        
        b_f;
        x_f;
        x_fn;
        x_sg;
        x_sgn;
        z_sg;
        z_fg;
    end
    methods
        function F=Fluid(dt,dz,m_e)
            F.dt=dt;
            F.dz=dz;
            F.m_e=m_e;
            F.m_fo=6+F.m_e;
            F.m_fg=F.m_e;
            F.m_fh=F.m_e;
            F.m_sg=F.m_e;
            F.ell=F.dz*F.m_e;
            F.C_d=0.15*F.mltom3/F.mmHgtoPa;
            F.C_po=1.45*F.mltom3/F.mmHgtoPa;
            F.C_p=F.C_po/(5.0/4.0);
            F.R=1.0*F.mmHgtoPa/F.mltom3;
            F.L=0.025*F.mmHgtoPa/F.mltom3;
            F.p_v=-60.0*F.mmHgtoPa;
            F.SV=80.0*F.mltom3;
            F.T_b=1.0/F.HR;
            F.T_s=sqrt(F.T_b)/3.0;
            F.QA=-6.0*F.SV/F.T_s^3;
            F.QB=6.0*F.SV/F.T_s^2;
            
            F.C_f=(2.0*F.dz)/(F.dt*F.r_o)*eye(F.m_fh,F.m_sg);
            F.D_f=F.C_f;
            F.M_f=zeros(F.m_fo+F.m_fg);
            F.M_f(1,1)=1.0;
            F.M_f(2,2)=1.0;
            F.M_f(3,3)=1.0;
            F.M_f(4,4)=F.C_p/F.dt;
            F.M_f(4,5)=1.0;
            F.M_f(5,5)=-1.0;
            F.M_f(5,6)=F.C_d/F.dt+1.0/F.R;
            F.M_f(6,4)=-1.0/F.sca;
            F.M_f(6,5)=F.L/F.dt/F.sca;
            F.M_f(6,6)=1.0/F.sca;
            F.M_f(3,F.m_fo-1)=1.0;
            F.M_f(3,F.m_fo)=-2.0;
            F.M_f(4,3)=-pi*F.r_o^2;
            F.M_f(2,F.m_fo+1)=-2.0;
            F.M_f(2,F.m_fo+2)=1.0;
            F.M_f(7,2)=-0.5/F.rho;
            F.M_f(F.m_fo,4)=0.5/F.rho;
            F.M_f(7:F.m_fo,7:F.m_fo)=F.dz/F.dt*eye(F.m_fo-6);
            F.M_f(7:F.m_fo,F.m_fo+1:F.m_fo+F.m_fg)=0.5/F.rho...
                *(diag(ones(F.m_fg-1,1),1)-diag(ones(F.m_fg-1,1),-1));
            F.M_f(F.m_fo+1,1)=-0.5;
            F.M_f(F.m_fo+1,2)=-F.dt/F.dz/F.rho;
            F.M_f(F.m_fo+F.m_fg,3)=0.5;
            F.M_f(F.m_fo+F.m_fg,4)=-F.dt/F.dz/F.rho;
            F.M_f(F.m_fo+1:F.m_fo+F.m_fg,7:F.m_fo)=0.5...
                *(diag(ones(F.m_fo-7,1),1)-diag(ones(F.m_fo-7,1),-1));
            F.M_f(F.m_fo+1:F.m_fo+F.m_fg,F.m_fo+1:F.m_fo+F.m_fg)...
                =F.dt/F.dz/F.rho*(2.0*eye(F.m_fg)...
                -diag(ones(F.m_fg-1,1),1)-diag(ones(F.m_fg-1,1),-1));
            F.N_f=zeros(F.m_fo+F.m_fg);
            F.N_f(4,4)=F.C_p/F.dt;
            F.N_f(5,6)=F.C_d/F.dt;
            F.N_f(6,5)=F.L/F.dt/F.sca;
            F.N_f(7:F.m_fo,7:F.m_fo)=F.dz/F.dt*eye(F.m_fo-6);
            
            F.n=0;
            
            F.x_f=zeros(F.m_fo+F.m_fg,1);
            F.x_fn=F.x_f;
            F.x_sg=zeros(F.m_sg,1);
            F.x_sgn=F.x_sg;
            F.z_sg=dz/2:dz:m_e*dz;
            F.z_fg=F.z_sg;
        end
        function x_fg=calculate_primal(F,x_sg)
            F.x_sg=x_sg;
            F.x_f=F.M_f\(F.b_f+F.N_f*F.x_fn...
                +[zeros(F.m_fo+F.m_fg-F.m_fh,1); F.D_f*F.x_sgn]...
                -[zeros(F.m_fo+F.m_fg-F.m_fh,1); F.C_f*F.x_sg]);
            x_fg=F.x_f(F.m_fo+1:F.m_fo+F.m_fg,1);
        end
        function v=inlet_velocity(F)
            t=F.n*F.dt;
            if (mod(t,F.T_b)<F.T_s)
                q=F.QA*(mod(t,F.T_b))^2+F.QB*mod(t,F.T_b);
            else
                q=0.0;
            end
            v=q/pi/F.r_o^2;
        end
        function increase_time(F)
            F.n=F.n+1;
            
            F.b_f=zeros(F.m_fo+F.m_fg,1);
            F.b_f(1,1)=F.inlet_velocity();
            F.b_f(5,1)=F.p_v/F.R;
            F.x_fn=F.x_f;
            F.x_sgn=F.x_sg;
        end
    end
end