% Copyright (c) 2011 Joris Degroote
%
% Licensed under the Non-Profit Open Software License version 3.0
% (NPOSL-3.0). You should have received a copy of the NPOSL-3.0 along with 
% this file. If not, see <http://www.opensource.org/licenses/NPOSL-3.0>
%
% Fluid-structure interaction simulation using IQN-ILS coupling.
%
% Reference: J. Degroote, K.-J. Bathe, and J. Vierendeels. Performance of a
% new partitioned procedure versus a monolithic procedure in fluid-
% structure interaction. Computers & Structures, 87(11�12):793�801, 2009. 
%
% DOI: 10.1016/j.compstruc.2008.11.013

function Simulation(graphics,output)

if (nargin<1)
    graphics=true;
end
if (nargin<2)
    output=2;
end
k_min=2;
k_e=10;
m_e=100;
n_e=100;
ell=0.06;
dt=0.01;
dz=ell/m_e;
epsilon=1e-5;
small=1.0e2*eps;
omega=0.01;

fluid=Fluid(dt,dz,m_e);
structure=Structure(dt,dz,m_e);
animator=Animator(dt,fluid.x_sg,fluid.z_sg,n_e*dt,10.0*dt,graphics);
convergence=Convergence(epsilon,k_min,small,output);
extrapolator=Extrapolator(fluid.x_sg);
model=Model(small);
objects={fluid,structure,animator,convergence,extrapolator,model};

for n=1:n_e
    cellfun(@increase_time,objects);
    
    for k=1:k_e
        if (k==1)
            x=extrapolator.predict();
        elseif (k==2)
            x=x+omega*r;
        else
            dx=model.predict(-r)+r;
            x=x+dx;
        end
        y=fluid.calculate_primal(x);
        xt=structure.calculate_primal(y);
        r=xt-x;
        model.add(r,xt);
        
        convergence.add(r);
        if (convergence.is_satisfied())
            break;
        end
    end
    animator.add(fluid.x_sg,fluid.z_sg);
    extrapolator.add(x);
end;